## BOSCH IoT INSIGHTS - Processing Pipeline - Embedded Linux Haskell Example

This example uses a small Haskell program that is compiled (with option use shared libraries) in a multi-stage 
Dockerfile. The Dockerfile uses a separate base image that contains the Haskell compiler ghc. It installs 
some additional libraries for json processing (aeson) and utf-8-strings.

The build image (`FROM docker.io/haskell:slim-buster as build`) compiles the executable and copies all
necessary libraries into a specific directory (`/opt/lib`). As a next step the image that should be used inside
Insights' pipeline processing is built (`FROM docker.io/ubuntu:23.10`). It must contain `fakechroot` 
and the compiled Haskell application. The necessary libraries are copied from the build-stage image to the
run-stage image. Inside the run-stage image, the application is located in `/opt/myapp/Main` 
and the additional necessary libraries are located in `/opt/lib`.

### Build the haskell_image with its multi-stage Dockerfile
```sh
docker build -t haskell_image -f ./resources/Dockerfile .
```

### Create the image which includes building the file system for the image
```sh
CONTAINER_ID=$(docker create haskell_image) && echo $CONTAINER_ID
```

### Export the file system from the image and compress it as xz-file
```sh
docker export $CONTAINER_ID | xz > ./resources/distro_flat.xz
```

### Adapt the python example to the different environment (ubuntu, Haskell executable)

The environment variables for this example (because we are using Ubuntu:23:10 as a base image) are as follows:
The `FAKEROOT_ELFLOADER` uses another loader (`ld-linux-x86-64.so.2`) than the Python example does. And the `LIBRARY_PATH`
must be adapted with the directory containing the additional libraries (`$(pwd)/opt/lib/`). 

```sh
FAKECHROOT_ELFLOADER=$(pwd)/usr/lib/x86_64-linux-gnu/ld-linux-x86-64.so.2
FAKECHROOT_BASE=$(pwd) 
LD_PRELOAD=$(pwd)/usr/lib/x86_64-linux-gnu/fakechroot/libfakechroot.so 
LD_LIBRARY_PATH=$(pwd)/usr/local/lib:$(pwd)/usr/local/lib/x86_64-linux-gnu:$(pwd)/lib/x86_64-linux-gnu:$(pwd)/usr/lib/x86_64-linux-gnu:$(pwd)/opt/lib
$(pwd)/usr/lib/x86_64-linux-gnu/ld-linux-x86-64.so.2 $(pwd)/opt/myapp/Main
```

```sh
docker run --rm -it -v "/$(pwd)/../../scripts:/scripts" haskell_image bash -c "./scripts/create_environment.sh"
```

The `constant.py` for this example must contain the modified path to the loader library from Ubuntu.
It must also contain the directory with the additional libraries which was copied into `/opt/lib` for
demonstration purpose. The path to `/opt/lib` is added to the LIBRARY_PATHS and will be used as value
for the `LD_LIBRARY_PATH` environment variable at runtime.

```python
# Paths for the specific loader library from your own filesystem must be adapted
OWN_ELF_LOADER = "$(pwd)/usr/lib/x86_64-linux-gnu/ld-linux-x86-64.so.2"
# Path to the required and installed fakechroot library must be adapted
OWN_PRELOAD = "$(pwd)/usr/lib/x86_64-linux-gnu/fakechroot/libfakechroot.so"
# Paths to all the necessary used library paths in your own filesystem must be adapted
OWN_LIBRARY_PATHS = "$(pwd)/usr/local/lib:$(pwd)/usr/local/lib/x86_64-linux-gnu:" \
                    "$(pwd)/lib/x86_64-linux-gnu:$(pwd)/usr/lib/x86_64-linux-gnu:" \
                    "$(pwd)/opt/lib"

# Path to the extracted filesystem relative to the current working directory after startup
OWN_FILE_SYSTEM_DIR = "./distro"
```

### Calling the executable from a python step

The json input with payload is redirected into the input-stream (`$(pwd)/opt/myapp/Main <<< \'' + json.dumps(document['payload']) + '\'`) 
of the Haskell application. The output is read by `run_with_own_loader` method and appended to the payload in new 
field `haskell_data`

```python
async def invoke(self, exchange_data_holder: ExchangeDataHolder) -> ExchangeDataHolder:
    print("\nStart processing " + str(exchange_data_holder))

    for document in exchange_data_holder.documents:
        json_str = '{"payload":' + json.dumps(document['payload']) + ',"metaData":{} }'
        document['payload']['haskell_data'] = run_with_own_loader('$(pwd)/opt/myapp/Main <<< \'' + json_str + '\'')

        # error message and trace should show up in the Insights Console
        if (document.get('metaData') or {}).get('hasError', None):
            # pass
            raise Exception('My Error')
            # also possible is exchange_data_holder.error = "Error"

    print("processed %s\n" % exchange_data_holder.id)

    return exchange_data_holder
```

The input of payload could also be stored in a temp file. The name of the file should be unique. Otherwise, 
it might be overwritten from other processes processing data at the same time (concurrent processing).
A file could also be used as an input-stream for the executable like this: (`$(pwd)/opt/myapp/Main < file_path`).


### Package the zip file for this example
Run following commands to prepare the content of the folder to zip.
```sh
mkdir -p ../../target
cp -r ../../src ../../executable-manifest.yaml ./resources ./src ../../target
```

If you have `zip` installed, you can run the following:
```
cd ../../target && zip -r haskell-example.zip * && cd -
```

Otherwise, you can also create you zip archive manually by selecting **the content** of the `target` folder
(do not select the folder itself!) and adding it to a zip archive.

**Hint**: Modify the executable-manifest.yaml and increase the version if you modify the example and upload it 
once more.
