const insightsUrl = 'https://bosch-iot-insights.com';
const project = process.env.INSIGHTS_PROJECT ?? '';
const username = process.env.INSIGHTS_USER ?? '';
const password = process.env.INSIGHTS_PASS ?? '';

const deviceType = 'vehicle';
const deviceIds = ['vehicle_audi_1', 'vehicle_audi_2', 'vehicle_audi_3', 'vehicle_audi_4', 'vehicle_opel_1', 'vehicle_skoda_1', 'vehicle_vw_1'];

function initDevicesData(devices) {
  const payload = {
    timestamp: new Date().toISOString(),
    position: {
      lat: 51.054627,
      long: 9.17405,
    },
    speed: 0,
    fuel: 100,
    odometer: 10,
    errors: [],
  };
  const dataByDevice = [];
  for (let i = 0; i < devices; i++) {
    const device = {
      id: deviceIds[i % deviceIds.length],
      type: deviceType,
      payload: { ...payload },
    };
    device.payload.position = { ...payload.position };
    dataByDevice.push(device);
  }
  return dataByDevice;
}

function* generateDeviceData(devices = 2) {
  const dataByDevice = initDevicesData(devices);

  for (let runs = 0; runs < Infinity; runs++) {
    let generated = false;
    for (const device of dataByDevice) {
      const { payload } = device;
      if (payload.fuel === 0) {
        console.log(`Device ${device.id} has run out of fuel.`);
        continue; // Stop generating data for this device
      }
      payload.timestamp = new Date().toISOString();
      payload.position.long += (Math.random() - 0.5) * 0.0001; // Simulate slight movement
      payload.position.lat += (Math.random() - 0.5) * 0.0001; // Simulate slight movement
      payload.speed += Math.max(0, Math.random() * 5 * (Math.random() > 0.5 ? -1 : 1)); // Random speed change
      payload.odometer += Math.random() * 1.5; // Random odometer increment
      payload.fuel = Math.max(0, payload.fuel - Math.random() * 0.9); // Decrease fuel slightly
      payload.errors = [];

      if (payload.fuel < 10) {
        payload.errors.push('Low fuel warning');
      }
      if (Math.random() < 0.2) {
        payload.errors.push('Engine error detected');
      }
      yield device;
      generated = true;
    }
    if (!generated) {
      console.log('All devices have run out of fuel. Stopping data generation.');
      break; // Stop the generator if all devices are out of fuel
    }
  }
}

async function sendData(data) {
  console.log('Generated Device Data:', data);
  const b64Credentials = btoa(`${username}:${password}`);
  await fetch(`${insightsUrl}/data-recorder-service/v2/${project}`, {
    method: 'POST',
    headers: {
      'Content-Type': 'application/json',
      'X-Metadata': JSON.stringify({
        device: data.id,
        thingId: project.replace('-', '_') + ':' + data.id,
        generator: 'device-simulator',
      }),
      Authorization: `Basic ${b64Credentials}`,
    },
    body: JSON.stringify(data.payload),
  });
}

async function generateAndSendDeviceData() {
  const deviceDataGenerator = generateDeviceData(7);
  while (true) {
    const data = deviceDataGenerator.next().value;
    if (data) {
      await sendData(data);
      await new Promise((resolve) => setTimeout(resolve, 3000)); // Simulate delay
    }
  }
}

if (!project || !username || !password) {
  console.error('Please set INSIGHTS_PROJECT, INSIGHTS_USER, and INSIGHTS_PASS environment variables.');
  process.exit(1);
}

if (process.env.http_proxy) {
  const undici = require('undici');
  // Set the ProxyAgent as global dispatcher for all fetch requests
  undici.setGlobalDispatcher(new undici.ProxyAgent(process.env.http_proxy));
}

generateAndSendDeviceData();
