#!/usr/bin/python3
import argparse
from base64 import b64encode
import glob
import itertools
import os
import subprocess
import sys
from threading import Thread
import traceback

import orjson


# the default place for the steps entry python script:
script = "src/step.py"

# common command line parameters
common_arguments_parser = argparse.ArgumentParser(add_help=False, fromfile_prefix_chars="@")
common_arguments_parser.add_argument("-w", "--write-results", nargs="?", const="{}_out.json", help="save each result to output files. Default filename_template is '{}_out.json'.", metavar="filename_template")
common_arguments_parser.add_argument("-l", "--line-format", action="store_true", help="disable transform resulting jsons in a more human readable way, getting them instead as a single compact line, e.g. to forward it to another step")
common_arguments_parser.add_argument("-q", "--quit", action="store_true", help="disables print out of results on stdout")

common_test_arguments_parser = argparse.ArgumentParser(add_help=False, fromfile_prefix_chars="@")
common_test_arguments_parser.add_argument("-s", "--step", nargs=1, help=f"script file to run (default: {script})", dest="script", default=[script], metavar="step_script")

parser = argparse.ArgumentParser(description='Locally test your step', fromfile_prefix_chars="@")

subparsers = parser.add_subparsers(help='', dest="command", metavar="command")

default_tests = subparsers.add_parser("default-tests", parents=[common_arguments_parser, common_test_arguments_parser], help="run the default test files (test/*.json)")

runfiles = subparsers.add_parser("run-files", parents=[common_arguments_parser, common_test_arguments_parser], help="run step and hand over the specified files")
runfiles.add_argument("files", nargs="+", help="input files", metavar="file_pattern")

flatten_file = subparsers.add_parser("flatten-files", help="removes all linebreaks in a (JSON) file")
flatten_file.add_argument("files", nargs="+", help="input files", metavar="file_pattern")

flatten_file = subparsers.add_parser("as-upload", parents=[common_arguments_parser], help="wraps given files into the input trigger format of IoT Insights")
flatten_file.add_argument("files", nargs="+", help="input files", metavar="file_pattern")

def validate_json_file(filename):
    try:
        with open(filename, encoding="utf-8") as file:
            orjson.loads(file.read())
    except BaseException as e:
        exit("not json: " + filename + " - " + str(e))
        
def iterate_file_content(files, jsonrequired = True):
    if len(files) < 1:
        exit("No input files given")
    if jsonrequired:
        for filename in files:
            validate_json_file(filename)
    
    #move yield into sub function to run input() on
    #call, not with first next() request
    def content_generator():
        for filename in files:
            with open(filename, "rb") as file:
                yield file.readlines()
    return content_generator()

class ErrorStreamLogger(Thread):
    def __init__(self, process):
        self.process = process
        Thread.__init__(self)
        self.start()
    def run(self):
        stream = self.process.stderr
        for line in stream:
            line = str(line, "utf-8")
            print("err: ", line, end="", file = sys.stderr, flush=True)

def run_step(step_py, lines_source):
    # to run python scripts from local dir and example-snippets with working "import insights_protocol':
    testpy_dir = os.path.dirname(os.path.realpath(sys.argv[0]))
    os.environ["PYTHONPATH"] = os.pathsep.join([os.getenv("PYTHONPATH",""), testpy_dir  + os.sep + "src"])
    step_process = subprocess.Popen(["python", "-u", *step_py], 
                     stdin = subprocess.PIPE,
                     stdout = subprocess.PIPE,
                     stderr = subprocess.PIPE
                     )
    process_input_channel = step_process.stdin
    try:
        ErrorStreamLogger(step_process)
        for line in lines_source:
            process_input_channel.write(line)
            process_input_channel.flush()
            result = step_process.stdout.readline()
            yield result
    finally:
        process_input_channel.close()
        try:
            result_code = step_process.wait(5)
            if result_code:
                print("warning: result code =", result_code, file=sys.stderr)
        except subprocess.TimeoutExpired:
            print("warning: step didn't stop in time --> killing")
            step_process.kill()

def json_oneline_generator(file_contents):
    for content in file_contents:
        yield (b"".join( map(bytes.strip, content) ) + b"\n")
        
def binary_oneline_generator(file_contents):
    for content in file_contents:
        yield (b"".join( content ) + b"\n")
        
def line_combiner(file_contents):
    for content in file_contents:
        yield (b"\n".join( content ) + b"\n")

def beautify_json(lines):
    for line in lines:
        try:
            yield orjson.dumps( orjson.loads(line), option=orjson.OPT_INDENT_2 ) + b"\n"
        except:
            yield line

def print_all(line, file=os.fdopen(sys.stdout.buffer.fileno(), "wb")):
    file.write(line)
    file.flush()
        
def tee(lines, second_out):
    for line in lines:
        try:
            second_out(line)
        except:
            print(traceback.format_exc(), file=sys.stderr)
        finally: 
            yield line
        
def write_to_files(results, template="{}_out.json"):
    i=0
    for content in results:
        with open(template.format(i), "wb") as file:
            try:
                file.write(content)
                i+=1
            except:
                print(traceback.format_exc(), file=sys.stderr)

def glob_file_args(file_globs):
    pattern_arg_to_files = { pattern: list(glob.glob(pattern)) for pattern in file_globs }
    resolved_empty = [ pattern for (pattern, found_files) in pattern_arg_to_files.items() if not found_files ]
    if resolved_empty:
        print("warning: no files found for '" + "', '".join(resolved_empty) + "'", file = sys.stderr)
    return list( itertools.chain.from_iterable( pattern_arg_to_files.values() ) )
    

def run_files(args):
    if not args.files:
        parser.error("missing one or more file arguments")
        
    if args.write_results and not "{}" in args.write_results:
        args.files += [args.write_results]
        args.write_results = "{}_out.json"
    args.files = glob_file_args(args.files)
    if not args.files:
        print("no files found, nothing to do", file=sys.stderr)
        exit(0)
    
    file_content_iterator = iterate_file_content(args.files)
    file_content_iterator = json_oneline_generator(file_content_iterator)
    output = run_step(args.script, file_content_iterator)
    if not args.line_format:
        output = beautify_json(output)
    if not args.quit:
        output = tee(output, print_all)
    if args.write_results:
        write_to_files(output, args.write_results)
    else:
        for out in output: pass # we have to iterate, otherwise nothing would happen

def default_tests(args):
    args.files = glob.glob("./tests/*.json")
    run_files(args)
    
def flatten_files(args):
    if not args.files:
        parser.error("missing one or more file arguments")
    
    args.files = glob_file_args(args.files)
    file_content_iterator = iterate_file_content(args.files)
    file_content_iterator = json_oneline_generator(file_content_iterator)
    for content_as_one_line in file_content_iterator:
        print_all(content_as_one_line)

def fill_upload_template(content_generator):
    for content_as_one_line in content_generator:
        size = len(content_as_one_line)
        base64 = str(b64encode(content_as_one_line),"ascii")
        wrapped = ( 
r'''{
    "_id": {
        "collection": "exampleproj_input_data",
        "document": { "$oid": "da7ac001da7ac001da7ac001" },
        "pipelineId": "exampleproj_pipeline",
        "projectId": "exampleproj"
    },
    "documents": [
        {
            "_id": { "$oid": "da7ac001da7ac001da7ac001" },
            "sourceCollection": "exampleproj_input_data",
            "contentType": "application/json",
            "receivedAt": { "$date": "2020-02-14T14:23:31Z" },
            "sender": {
                "user" : { 
                    "userId" : "00000000-0000-0000-0000-000000000000",
                    "userName": "exampleprj_user",
                    "tenantId": "00000000-0000-0000-0000-000000000000",
                    "tenantName": "exampleprj_tenant"
                }
            },
            "size": ''' + str(size) + r''',
            "tags": [],
            "metaData": {
                "type": "condition_sensor",
                "device": "EBSD1419230013"
            },
            "payload": {
                "$binary": "''' + base64 + r'''",
                "$type": "00"
            }
        }
    ],
    "trigger": "test"
}''')
        yield bytes(wrapped,"utf-8").splitlines()

def as_upload(args):
    if not args.files:
        parser.error("missing one or more file arguments")
    args.files = glob_file_args(args.files)
    file_content_iterator = iterate_file_content(args.files, False)
    file_content_iterator = binary_oneline_generator(file_content_iterator)
    output = fill_upload_template(file_content_iterator)
    if args.line_format:
        output = json_oneline_generator(output)
    else:
        output = line_combiner(output)
    if not args.quit:
        output = tee(output, print_all)
    if args.write_results:
        write_to_files(output, args.write_results)
    else:
        for out in output: pass # we have to iterate, otherwise nothing would happen   
    
def invalid_command(args):
    exit("unkown command " + args.tests)

commands = {
    "run-files": run_files,
    "default-tests": default_tests,
    "flatten-files": flatten_files,
    "as-upload": as_upload,
    }

if __name__ == "__main__":
    args = parser.parse_args()
    
    if not args.command:
        parser.error("No command given")
    (commands.get(args.command, invalid_command))(args)