#!/usr/bin/python3
from abc import ABC, abstractmethod
import asyncio
from asyncio.events import AbstractEventLoop
from collections import namedtuple, OrderedDict
from concurrent.futures.thread import ThreadPoolExecutor
from io import IOBase
import json
import os
import sys
import traceback

from bson import json_util
from bson.json_util import default as extjson_conversion, RELAXED_JSON_OPTIONS

# data model declerations - actual implementations below
class ExchangeDataHolder: pass 
class ExchangeDataHolderId: pass
class MongoDBRef: pass

# main class
class StepBase(ABC):
    """Example skeleton to process documents from stdin to stdout
      
    To use, inherit from StepBase and implement the invoke method.
    
    The process_stream() method will run the processing, so it is
    typically called in the programs main thread.
     
    The skeleton works asynchronous and non-blocking, so multiple
    documents can be processed almost parallel, e.g. if they do
    IO or make HTTP requests (recall CPython's limitiations
    regarding parallelity, described in 
    <https://docs.python.org/3/glossary.html#term-global-interpreter-lock>
    and <https://wiki.python.org/moin/GlobalInterpreterLock>). 
    """
    def __init__(self, input: IOBase, output: IOBase, loop: AbstractEventLoop = asyncio.get_event_loop()):
        self.input_channel = os.fdopen(input.fileno(), "r", encoding="utf-8", errors='ignore', newline="\n")
        self.output_channel = os.fdopen(output.fileno(), "w", encoding="utf-8", newline="\n")
        self.event_loop = loop
    
    @abstractmethod
    async def invoke(data_holder, exchange_data: ExchangeDataHolder) -> ExchangeDataHolder:
        pass

    def process_stream(self):
        self.event_loop.run_until_complete( self.__process_stream() )

    async def __process_stream(self):
        stdin_executor = ThreadPoolExecutor(max_workers=1)
        while True:
            json_line = await self.event_loop.run_in_executor(stdin_executor, self.input_channel.readline)
            if not json_line: # readline signal for end of file
                break # no more input -> exit process stream
            self.event_loop.create_task( self.__process_document(json_line) ) #run async
        await self.__wait_for_pending_tasks()

    async def __process_document(self, json_line: str):
        try:
            input = from_json( json_line )
            try:
                output = await self.invoke(input) or input
                if output.id != input.id:
                    raise ValueError("illegal 'id' returned in " + output)
                print(to_json(output), file=self.output_channel, flush=True)
            except:
                output = input._replace(documents=[], outputRef=[], variables={}, error=traceback.format_exc())
                print(to_json(output), file=self.output_channel, flush=True)
        except:
            print(traceback.format_exc())

    async def __wait_for_pending_tasks(self):
        this_task = asyncio.Task.current_task( self.event_loop )
        loop_tasks = asyncio.Task.all_tasks( self.event_loop )
        pending = [t for t in loop_tasks if t != this_task and not t.done()]
        await asyncio.gather(*pending, loop=self.event_loop, return_exceptions=True)


class ExchangeDataHolderId:
    # fields the protocol grants to exist:
    granted_fields_defaults = {'collection': None, 'document': None, 'pipelineId': None, 'projectId': None, }
    
    def __init__(self, doc: dict): 
        self.__dict__.update( {**self.granted_fields_defaults, **doc} )
        
    def __setattr__(self, name, value): 
        """ modifications would break insights protocol, so disallow """
        raise AttributeError(self.__class__.__name__ + " is immutable")
    
    def __repr__(self) -> str: return "%s(%r)"  % (self.__class__.__name__, self.__dict__)
    
    def to_extjson(self) -> dict: return self.__dict__
    
class MongoDBRef(namedtuple('MongoDBRef', ['id', 'collection'])):
    def to_extjson(self): return rename_id_in_dict( self._asdict() )

class ExchangeDataHolder:
    modifiable_fields = ('error')
    granted_fields_defaults = {
        'id': ExchangeDataHolderId( {} ),
        'documents': [], 
        'outputRef': [], 
        'trigger': "unknown", 
        'variables': {},
        'error': None,
    }
    def __init__(self, doc: dict):
        self.__dict__.update({
            **self.granted_fields_defaults,
            'id': ExchangeDataHolderId( doc.pop("_id", {}) ),
            **doc
        })
        
    def __setattr__(self, name, value):
        """ to avoid protocol violations, instance should be threaded like immutable, except special fields. """
        if not name in self.modifiable_fields:
            raise AttributeError(self.__class__.__name__ + " is immutable")
        self.__dict__[name] = value
        
    def _replace(self, **values):
        """ copy instance with **values overwriting attributes, like namedtuple supports it """
        return self.__class__({**self.__dict__, **values})
    
    def __repr__(self) -> str:
        return "%s(%r)" % (self.__class__.__name__, self.__dict__)
    
    def __str__(self) -> str:
        return repr({ 
            "id": self.id,
            "trigger": self.trigger,
             **( {'docs': len(self.documents)} if not self.error else {'error':self.error} ),
             **( {'oRef': len(self.outputRef)} if self.outputRef and not self.error else {} ),
             **( {'vars': len(self.variables)} if self.variables and not self.error else {} ),
        })
    
    def to_extjson(self) -> dict: return rename_id_in_dict( self.__dict__ )

def rename_id_in_dict(objdict: dict) -> OrderedDict:
    """rename id in dict, but keep order. Required for encoding to Enhanced JSON"""
    as_list_of_list = list(map(list, objdict.items()))
    as_list_of_list[0] = ['_id', objdict['id']]
    return OrderedDict(as_list_of_list)

def to_json(data_holder: ExchangeDataHolder) -> str:
    class JSONEncoder(json.JSONEncoder):
        """ specialize to handle objects with 'to_extjson' methods or named tuples with '_asdict' method """

        def encode(self, obj):
            if isinstance(obj, (bool, str)):
                return json.JSONEncoder.encode(self, obj)
            if hasattr(obj, 'to_extjson'):
                obj = obj.to_extjson()
            if hasattr(obj, '_asdict'):
                obj = obj._asdict()
            if isinstance(obj, dict):
                return "{" + ",".join([self.encode(k) + ':' + self.encode(v) for k, v in obj.items()]) + "}"
            elif isinstance(obj, (list, set, tuple)):
                return "[" + ",".join([self.encode(v) for v in obj]) + "]"
            else:
                try:
                    obj = extjson_conversion(obj, RELAXED_JSON_OPTIONS)
                    return self.encode(obj)
                except TypeError:
                    return json.JSONEncoder.encode(self, obj)

    return json.dumps(data_holder, ensure_ascii=False, cls=JSONEncoder)

def from_json(json) -> ExchangeDataHolder:
    return ExchangeDataHolder( json_util.loads(json) )
