import datetime
import json
from zipfile import ZipFile
import yaml
import os
import requests
import urllib.parse
import argparse

# Default values for command line parameters
# Normally this does not have to be changed
source_folder = 'src'
# Normally this does not have to be changed
endpoint = os.getenv('INSIGHTS_ENDPOINT', 'https://bosch-iot-insights.com')
# Has to be changed or overwritten by command line parameters:
project = os.getenv('INSIGHTS_PROJECT', 'PROJECT-NAME')
# Has to be changed or overwritten by command line parameters (user requires project ADMIN role):
project_user = os.getenv('INSIGHTS_PROJECT_USER', 'TECHNICAL_ADMIN_USER')
# Has to be changed or overwritten by command line parameters:
project_user_password = os.getenv('INSIGHTS_PROJECT_USER_PASSWORD', 'PASSWORD')


# Command line parameters
parser = argparse.ArgumentParser(description='Deploy python custom code to Insights pipeline.',
                                 epilog="Use the command line parameters or change default value in the script to change the credentials.")
parser.add_argument('-r', '--requirements', nargs='?', type=str, help='path to own requirements.txt file to overwrite auto dependency resolution')
parser.add_argument('-H', '--host', nargs='?', const=endpoint, default=endpoint, type=str, help='target url')
parser.add_argument('-p', '--project', nargs='?', const=project, default=project, type=str, help='project name')
parser.add_argument('-u', '--user', nargs='?', const=project_user, default=project_user, type=str, help='project user (user requires project ADMIN role)')
parser.add_argument('-P', '--password', nargs='?', const=project_user_password, default=project_user_password, type=str, help='project user password')
parser.add_argument('-f', '--folder', nargs='?', const=source_folder, default=source_folder, type=str, help='src folder with your python code '
                                                                                             '(all files in this folder will be added to the zip file)')
args = parser.parse_args()

artifacts_path = '/project-management-service/v1/'+args.project+'/pipeline/artifacts'

# Bump minor version
with open('executable-manifest.yaml') as file:
    manifest = yaml.full_load(file)

    query = {
        'metadata.manifest.name': manifest.get('name')
    }

    search_result = requests.get(args.host + artifacts_path + '?filterQuery=' + urllib.parse.quote(json.dumps(query)),
                                 auth=(project_user, project_user_password))
    if search_result.status_code != 200:
        search_result.raise_for_status()

    version_parts = str(manifest.get('version')).split('.')
    new_version = version_parts[0] + '.' + str(int(version_parts[1]) + 1)

    print('Current version', manifest.get('version'))
    print('New version', new_version)

    manifest['version'] = new_version

    if manifest['metadata']:
        manifest['metadata']['project'] = project
        # manifest['metadata']['creator'] = os.getenv('USERNAME', manifest['metadata']['creator'])
        if len(search_result.json()) == 0:
            print('Recognized as new artifact. Update createdAt.')
            manifest['metadata']['createdAt'] = datetime.datetime.utcnow().strftime("%Y-%m-%dT%H:%M:%SZ")
        manifest['metadata']['modifiedAt'] = datetime.datetime.utcnow().strftime("%Y-%m-%dT%H:%M:%SZ")

with open('executable-manifest.yaml', 'w') as file:
    yaml.dump(manifest, file, sort_keys=False)

# Pack zip file
with ZipFile('processor_artifact.zip', 'w') as zipObj:
    if os.path.exists('requirements.txt'):
        zipObj.write('requirements.txt')
    zipObj.write('executable-manifest.yaml')
    exclude = ['__pycache__']
    for current_dir, dirs, files in os.walk(args.folder):
        dirs[:] = [d for d in dirs if d not in exclude]
        for file_name in files:
            zipObj.write(os.path.join(current_dir, file_name))
    if args.requirements is not None:
        zipObj.write(os.path.join('.', args.requirements))

# Upload artifact
response = requests.post(args.host + artifacts_path,
                         auth=(args.user, args.password),
                         files={'file': open('processor_artifact.zip', 'rb')})
if response.status_code == 201:
    print('successful')
else:
    parser.print_help()
    response.raise_for_status()
