import glob
import os
import subprocess
import sys

import orjson  # using orjson instead of just json has several advantages

# more at https://github.com/ijl/orjson

# find all png files in the test_directory
test_directory = "tests"  # change the directory name here if needed
file_pattern = os.path.join(test_directory, "*.png")

# start up step.py - it will run as subprocess and will accept data the whole time until the script stops
path_to_python = sys.executable
step = subprocess.Popen(
    [path_to_python, "-u", "src/step.py"], stdin=subprocess.PIPE, stdout=subprocess.PIPE
)
# sys.executable contains the path to the current python interpreter (we want to use this one to run step.py)
# step.stdin (io.BufferedIOBase) accepts exchange_data json documents, one per line
# step.stdout (io.BufferedIOBase) will then contain a result line for each input line with the new exchange_data (as json)


def calculate_directory_stats():
    predictions = [{"total": 0, "correct": 0} for i in range(10)]
    overall = {"total": 0, "correct": 0}

    # call test.py to convert the png filenames to json
    output = subprocess.run(
        [path_to_python, "test.py", "as-upload", "-l", file_pattern],
        capture_output=True,
    )
    # python program "test.py" - several test utilities that simulate processes in Bosch IoT Insights pipelining
    # command "as-upload"      - wraps given files into the input trigger format of IoT Insights
    # parameter "-l"           - disables transformation of resulting jsons into human readable format with
    #                            line breaks, getting the whole json result as a single compact line instead,
    #                            which the script can forward to step.py

    # do the same glob as "test.py as-upload" does to retrieve filenames
    filenames = glob.glob(file_pattern)

    # output.stdout is a byte string with json documents wrapping each image
    step_in = output.stdout.splitlines()
    step_out = []
    # call step.py to get a prediction for all the images
    for n, doc_in in enumerate(step_in):
        input_file = filenames[n]
        step.stdin.write(doc_in)
        # insights protocol is line based:
        step.stdin.write(b"\n")
        # force writing to subprocess now - otherwise python would delay actual writing sometimes
        #                                   caused by internal buffering
        step.stdin.flush()
        # read result from step
        doc_out = step.stdout.readline()
        # collect all documents in step_out for later processing
        step_out.append({"input_file": input_file, "output_json": doc_out})

    # go through output of each result
    for single_result in step_out:
        # parse byte string to dictionary using a json parser
        file = single_result["input_file"]
        json = single_result["output_json"]

        json_parsed = orjson.loads(json)
        prediction = json_parsed["documents"][0]["metaData"]["prediction"][0]

        predicted_digit = prediction.index(max(prediction))
        filename = os.path.basename(file)
        true_digit = int(filename[0])

        predictions[true_digit]["total"] += 1
        is_correct = true_digit == predicted_digit
        if is_correct:
            predictions[true_digit]["correct"] += 1

    # compute statistics of correct predictions per digit
    for digit, digit_stats in enumerate(predictions):
        percentage_correct = (
            round(digit_stats["correct"] / digit_stats["total"] * 100)
            if digit_stats["total"] > 0
            else 0
        )
        print(
            f"digit {digit}: correct: {digit_stats['correct']:5d} of {digit_stats['total']:5d} ({percentage_correct:3d}%)"
        )
        overall["correct"] += digit_stats["correct"]
        overall["total"] += digit_stats["total"]

    average_percentage_correct = (
        round(overall["correct"] / overall["total"] * 100)
        if overall["total"] > 0
        else 0
    )
    print("---------------------------------------")
    print(
        f"average: correct: {overall['correct']:5d} of {overall['total']:5d} ({average_percentage_correct:3d}%)"
    )


def react_on_any_change_in_directory():
    """
    waiting for changes in target directory, triggers stats recalculation
    on each change

    uses watchdog - which works on different OS, incl. Mac, Windows, Linux
    see https://github.com/gorakhargosh/watchdog
    """
    import watchdog.events
    import watchdog.observers

    class DirectoryEventHandler(watchdog.events.FileSystemEventHandler):
        """watchdog event handler which just triggers recalculation on every event"""

        def on_any_event(self, event):
            calculate_directory_stats()

    directory_observer = watchdog.observers.Observer()
    directory_observer.schedule(DirectoryEventHandler(), test_directory)
    directory_observer.start()
    try:
        while directory_observer.is_alive():
            directory_observer.join(1)
    finally:
        directory_observer.stop()
        directory_observer.join()


# calculate statistics of the directory for the first time
calculate_directory_stats()
# now, recalculate stats with every change in that directory
# react_on_any_change_in_directory()
